<?php

defined('_JEXEC') or die;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri as JUri;

/**
 * Quix Asset Manager for managing CSS and JavaScript assets
 * Supports both legacy Joomla versions and modern Web Asset Manager
 *
 * @package     QuixNxt\Component\Quix
 * @subpackage  Administrator\Helper
 * @since       1.0.0
 * @copyright   Copyright (C) 2024 QuixNxt. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */
class QuixHelperAsset
{
  public $scripts  = array();
  public $styles   = array();
  private $extension = 'quixnxt';
  private $version = QUIXNXT_VERSION;
  private $document;
  private $assets_json;

  public function __construct()
  {
    $is_legacy_mode = defined('JVERSION') ? version_compare(JVERSION, '4.3', '<') : true;
    $quix_assets_json_path = JPATH_SITE . '/media/' . $this->extension. '/joomla.asset.json';
    $this->assets_json = json_decode(file_get_contents($quix_assets_json_path), true);

    if ($is_legacy_mode) {
      $this->document = Factory::getDocument();
    } else {
      $this->document = Factory::getApplication()->getDocument();
    }
  }

  /**
   * Load all assets (CSS, JS, and admin script) based on Joomla version
   * @since 5.2.10
   * @param array $cssAssets    CSS assets array: ['asset.name' => '/css/file.css']
   * @param array $scriptAssets Script assets array: ['asset.name' => '/js/file.js']
   */
  public function loadAssets(array $cssAssets = [], array $scriptAssets = [])
  {
    if ($this->isLegacyMode()) {
      $this->loadLegacyStyles($cssAssets);
      $this->loadLegacyScripts($scriptAssets);
    } else {
      $this->loadWebStyles($cssAssets);
      $this->loadWebScripts($scriptAssets);
    }
  }

  public function loadStyles(array $cssAssets)
  {
    if ($this->isLegacyMode()) {
      $this->loadLegacyStyles($cssAssets);
    } else {
      $this->loadWebStyles($cssAssets);
    }
  }

  public function loadScripts(array $scriptAssets)
  {
    if ($this->isLegacyMode()) {
      $this->loadLegacyScripts($scriptAssets);
    } else {
      $this->loadWebScripts($scriptAssets);
    }
  }

  private function loadWebScripts(array $scriptAssets)
  {
    $wam = $this->document->getWebAssetManager();
    $wam->getRegistry()->addExtensionRegistryFile($this->extension);
    $handle_manual = !QUIXNXT_LOAD_ABSOLUTE_URL;

    foreach ($scriptAssets as $assetKey => $_) {
      if ($handle_manual) {
        $asset = $this->getAssetFromJson($assetKey);
        $uri = $asset['uri'];
        $attributes = $asset['attributes'];
        $path = $this->getAssetUrl($uri, 'js');
        $wam->registerAndUseScript(
          $assetKey,
          $path,
          array('version' => $this->version),
          $attributes
        );
      } else {
        $wam->useScript($assetKey);
      }
    }
  }

  private function loadWebStyles(array $cssAssets)
  {
    $wam = $this->document->getWebAssetManager();
    $wam->getRegistry()->addExtensionRegistryFile($this->extension);
    $handle_manual = !QUIXNXT_LOAD_ABSOLUTE_URL;
    foreach ($cssAssets as $assetKey => $_) {
      if ($handle_manual) {
        $asset = $this->getAssetFromJson($assetKey);
        $uri = $asset['uri'];
        $path = $this->getAssetUrl($uri, 'css');
        $wam->registerAndUseStyle(
          $assetKey,
          $path,
          array('version' => $this->version)
        );
      } else {
        $wam->useStyle($assetKey);
      }
    }
  }

  public function loadAdminScript()
  {
    $scriptUrl = JUri::root() . '/administrator/components/com_quix/assets/script.js';

    if ($this->isLegacyMode()) {
      $this->document->addScript(
        JUri::root(true) . '/administrator/components/com_quix/assets/script.js',
        array('version' => $this->version),
        array('defer' => true)
      );
    } else {
      $wam = $this->document->getWebAssetManager();
      $wam->registerAndUseScript(
        $this->extension . '.administrator-script',
        $scriptUrl,
        array('version' => $this->version),
        array('defer' => true)
      );
    }
  }

  /**
   * Add inline JavaScript to the document
   *
   * @param string $script The JavaScript code to add
   * @param array $options Options for the script (position, etc.)
   * @since 5.2.10
   * @return void
   */
  public function inlineScript(string $script , array $options = [])
  {
      if ($this->isLegacyMode()) {
          $this->document->addScriptDeclaration($script);
      } else {
          // Modern mode - use Web Asset Manager
          $wam = $this->document->getWebAssetManager();
          $wam->addInlineScript($script, $options);
      }
  }

  public function isLegacyMode()
  {
    return defined('JVERSION') ? version_compare(JVERSION, '4.3', '<') : true;
  }

  private function loadLegacyStyles(array $cssAssets)
  {
    foreach ($cssAssets as $assetKey => $_) {
      $asset = $this->getAssetFromJson($assetKey);
      $uri = $asset['uri'];
      $path = $this->getAssetUrl($uri, 'css');
      $this->document->addStyleSheet($path, array('version' => $this->version));
    }
  }

  private function loadLegacyScripts(array $scriptAssets)
  {
    foreach ($scriptAssets as $assetKey => $_) {
      $asset = $this->getAssetFromJson($assetKey);
      $uri = $asset['uri'];
      $attributes = $asset['attributes'];
      $path = $this->getAssetUrl($uri, 'js');
      $this->document->addScript($path, array('version' => $this->version), $attributes);
    }
  }

  private function getAssetFromJson(string $key)
  {
    $assets = $this->assets_json["assets"];
    foreach ($assets as $asset) {
      if ($asset["name"] == $key) {
        return $asset;
      }
    }
    throw new Exception("Asset not found in assets.json");
  }

  private function getAssetUrl(string $uri, string $type): string
  {
    $ds = DIRECTORY_SEPARATOR;
    $parts = explode('/', $uri, 2);
    if (count($parts) < 2) {
      $parts = [$this->extension, $uri];
    }
    $path = $ds . 'media' . $ds . $parts[0] . $ds . $type . $ds . $parts[1];
    $real = realpath(JPATH_SITE . $path);
    $resolvedPath = $real ? str_replace(JPATH_SITE, '', $real) : $path;

    if (!QUIXNXT_LOAD_ABSOLUTE_URL) {
        $asset = str_replace($ds . "media" . $ds . "quixnxt", "", $resolvedPath);
        $asset_key = "quix-asset";
        return JUri::root() . "index.php?$asset_key=$asset&var=$this->version";
    }
    $urlPath = str_replace('\\', '/', $resolvedPath);
    return JUri::root(true) . $urlPath;
  }


}
